;********************************************************************************
;* dds32.asm
;* Control uC fr den DDS Generator nach Jesper Hansen 
;* http://www.myplace.nu/avr/minidds/index.htm
;* Bedienelement ist nur ein Drehgeber mit Schalter.
;* Der Schalter schaltet zwischen Kommando- und Ausfhrungsmodus um
;*
;* Um die Umschaltstrungen der DDS Signale kurz zu halten wird eine schnelle 
;* Kommunikation mit 500 Kbaud via RS232 angewendet. 
;* Im Steuer uC wird die Frequenz ausgerechnet und 5 Bytes an den DDS uC gesendet
;********************************************************************************

.include "tn2313def.inc"		;Prozessor ATTINY2313

;            ATtiny2313					Display TC1602A		4 bit Mode
;            +--+-+--+					2 Zeilen zu je 16 Stellen
;     !RESET |  |_|  | VCC				Memoryadr.: Zeile1: 00  und Zeile2: 40h
;    (RX)PD0 |       | PB7(SCK)			n.c.	
;    (TX)PD1 |       | PB6(MISO)		n.c.		
;      XTAL2 |       | PB5(MOSI)-----	Display pin 4 RS
;      XTAL1 |       | PB4		-----	Display pin 6 Enable	
;  (INT0)PD2 |       | PB3(OC1)	-----	Display pin 14
;  (INT1)PD3 |       | PB2		-----	Display pin 13	
;    (T0)PD4 |       | PB1(AIN1)-----	Display pin 12	
;    (T1)PD5 |       | PB0(AIN0)-----	Display pin 11	
;        GND |       | PD6				Display pin 1 Masse	
;            +-------+					Display pin 2 VCC+	
;										Display pin 3 Kontrast
; PD0	I	RX RS232					Display pin 5 R/W an Masse
; PD1	O	TX RS232					Display pin 15 Beleuchtung +
; PD2	O	Rotary Encoder phaseB		Display pin 16 Display pin Masse
; PD3	O	Rotary Encoder phaseA
; PD4	I	Switch von Encodertaste
; PD5	I	n.c.
; PD6	I	n.c.
				
.equ XTAL = 8000000           ;systemclock [Hz] 16 Mhz im DDS_uC
.equ BAUD  = 500000           ;Terminal Baudrate, double speed
 
; UBRR Wertfr RS232 ausrechnen
.equ UBRRL_VAL   = (XTAL/(BAUD*8)-1)  ; 

;*******************************************************************
;* Fuse setting auf 0 (gesetzt):
;* Ext. Byte:	Selfpgren enabled
;* Hi Byte:		Default
;* Low Byte:	"FF" = Clock undivided, startup long, >8MHz Quarz
;* ;*****************************************************************

;calculation variables
.def	var_form = r2	;Index auf Spannungsform
.def	var_step = r3	;Index auf Frequenzstepwert als auch Steptext

.def	RY0	= r4		;4 Byte Konstante fr Frequenzberechnung
.def	RY1	= r5		;16e6/11*10 
.def	RY2	= r6		; *10 weil ja 1/10 Hz Einheiten 
.def	RY3	= r7		;im Display angezeigt werden

.def	RR0	= r8		;Adderwert fr Phasenaccu 32 bit
.def	RR1	= r9		;max. 0x01c28f5c = 10 kHz Step
.def	RR2	= r10		
.def	RR3	= r11		;

.def	RR4	= r12		;der Phasenaccu 32 bit mit 0,6875 us je update
.def	RR5	= r13		;Max.Wert 0x12000000 entspricht 102272,7 Hz
.def	RR6	= r14
.def	RR7	= r15 

.def	RX0	= r16		; Resultat der Frequenzberechnung ist auch
.def	RX1	= r17		; gleich der Input fr die BCD Ausgabe
.def	RX2	= r18		;
.def	RX3	= r19		;
.def	RX4	= r20		; BCD value digits 0 and 1
.def	RX5	= r21		; BCD value digits 2 and 3
.def	RX6	= r22		; BCD value digits 4 and 5
.def	RX7	= r23		; BCD value digits 6 and 7 (MSD) 

;allgemeine Variablen  
.undef  XL
.undef  XH
.undef  YL
.undef	YH
.def	A 		= r24  	;allg. Verwendung
.def	B		= r25
.def	C		= r29	;allg Variable 
.def	neu		= r25	;doppelt belegt
.def	var		= r26	;(XL)
.def    old 	= r27	;(XH)old value of the encoders
.def	flag	= r28	;Flagregister fr Encoder und Taste

			.CSEG		; ab hier kommt Programmcode
.org 0x0000
        rjmp    init	; Reset Handler
.org 0x0006
        rjmp    Timer0	; Timer0 ISR alle ~2ms
.org 0x0007                    
		reti
 
.org	0x0013
_1cmd:	.db	"DDS    Parameter",00,00				;9 words
_1Zeile:.db	"DDS     +-",00,00;						;6 words
													;=$0F words
;Strings fr den Rest der 1. Zeile, bzw 2.Zeile Commandmode	
_01:	.db" 0,1Hz", 00,00	;Strings padded auf gerade Zeichenanzahl 
_1:		.db"   1Hz", 00,00	;je 4 Words
_10:	.db"  10Hz", 00,00
_100:	.db" 100Hz", 00,00
_1k:	.db"  1kHz", 00,00	;Initwert							
_10k:	.db" 10kHz", 00,00							
sin:	.db"Sinus  ", 00	;im DDS_uC sind die Tabellen in dieser					
tria:	.db"Dreieck", 00	;Reihenfolge angelegt
saw:	.db"Saegez.", 00
sq_r:	.db"Recht_k" ,00							;=$24 words

;Accuphasenstepwerte entsprechend den Strings oberhalb
;dem DDS-uC wird immer die Spannungsform und der Phasenaccuwert bergeben
;in Summe 5 Bytes

z104:	.db $00,$00,00,$12	;Maxwert DDSaccu entspricht 102272,7Hz
z555:	.db $2e,$f2,$dd,00	;Konstante fr die Frequenzausgabe = 16e6/11*10
							;Genauigkeit siehe ddsgen.asm
z01:	.db	$27,$01,$00,00	;0 ppm	
z1:		.db	$89,$0b,$00,00	;0 ppm	
z10:	.db	$58,$73,$00,00	;0 ppm					;je 2 words/Zahl
z100:	.db	$6f,$81,$04,00	;3 ppm						
z1k:	.db	$56,$0e,$2d,00	;57 ppm
z10k:	.db	$5c,$8f,$c2,01	;0,09%					;=$10 words
		.db"Tend"									;2 words
;In Summe  $13 +$0F +$28 +$10 +$2 =$5C words =  	$B8 Bytes 
;Adressierung der Werte ber z hat noch immer ZH = 0

;************************************************************
init:
;init stack 
        ldi     A,	 LOW(RAMEND)	;Stackpointer init
		out     SPL, A

;init Timer 0 interrupt alle 2,05 ms fr Encoderabfrage
        ldi     A,	 0b00000011		;set CS00 : divisor: 64
		;TCCR0A  ist Default 0
        out     TCCR0B, A			;Seite 80
        ldi     A,	 1 << TOIE0		;Interrupt at Timer Overflow
        out     TIMSK, A			;Seite 82

;init PortD Encoder
		ldi	A,	0b00000010	;nur pin PD1 (TX) ist output =1 
		out	DDRD, A			;sonst PortD ist input
		ldi     A, 	0xFF		;Pull-Up resistors on Port D
		out	PortD,	A		;bzw PD1 = 1
		clr	old			;default "old value" Encoder = no action

;Init serial Port (UART) "polling", double speed
		ldi     C, HIGH(UBRRL_VAL)	;set Baudrate = 1 = 500 kBaud
		out     UBRRH, C
		ldi     C, LOW(UBRRL_VAL)
		out     UBRRL,C
		ldi	C,2				;nur bei 8 MHz clock 
		out	UCSRA,C				;double speed = bit1 Seite 134
 		ldi	C,6				;Frame format 8N1, Seite137
		out     UCSRC,C				;an sich der Defaultwert
		ldi	C,8	
		out     UCSRB,C	        	;TX enable = bit3 Seite 135

;Init the LCD an PortB entsprechend TC1602A_Manual.pdf
lcd_init:
		ldi		C,$FF				;PortB alle auf Output Seite50
		out		DDRB,C
		;clr		C				;geht net wieso?
		out		PortB,C				;definierter Status frs Display
		ldi		B,100				;n x 2ms delay
wait_display:						;t2313 ist fertg, display auch
		rcall delay2ms				;lass alles zur Ruhe kommen
		dec   B
		brne  wait_display

; Display DB [7:4] liegt auf PortB [3:0]
; sonst knnte man nicht RS = PB5,  Enable = PB4 
; im 4 bit mode vom PortB aus bedienen
		ldi   C,0b00000011	;8bit mode, 
		out   LCD_PORTB, C	;
		rcall lcd_enable	;1,  
		rcall delay2ms
		rcall delay2ms
		rcall lcd_enable	;2
		rcall delay2ms
		rcall lcd_enable	;3
		rcall delay2ms		;Display ist jetzt eindeutig im 8bit mode

;LCD: function set 
		ldi   C,0b00000010;4bit-Mode, noch im 8 bit mode gesendet
		out   LCD_PORTB, C
		rcall lcd_enable		
		rcall delay2ms		;Display ist jetzt im 4bit mode	
				
; ab nun kann das Byte jeden Wert haben, da es in 2 nibbles aufgeteilt
; wird, die via PB[3:0] kommend, an das Display DB[7:4] gelangen.

funcset:ldi	C,	0b00101000	;4bit, 2 Zeilen, 5x7 dots + cursor
		rcall	lcd_cmd
disp_on:ldi	C,	0b00001111	;Display on, Cursor on, blink
		rcall	lcd_cmd
dispclr:ldi	C,	0b00000001	;clear Display,Cursor home,
		rcall	lcd_cmd		;executiontime 1,52 ms
		rcall	delay2ms	;deshalb delay
entmode:ldi	C,	0b00000110	;increment, Cursor move 
		rcall	lcd_cmd

;init Konstante z555 fr Frq. Berechnung in RY[3:0] und RX[3:0]DDSaccu
		clr		ZH
		ldi		ZL,z555 * 2
		lpm		RY0,Z+		;Faktor 16e6/11*10
		lpm		RY1,Z+
		lpm		RY2,Z+
		lpm		RY3,Z
		mov		RR0,ZH
		mov		RR1,ZH
		mov		RR2,ZH
		inc		RR2			;set 0,08638 Hz, sonst wrs ja DC
		mov		RR3,ZH
		mov		RR4,ZH
		mov		RR5,ZH
		mov		RX3,ZH
		mov		RX4,ZH	
		mov		RX5,ZH
		mov		RX6,ZH
		mov		RX7,ZH	

		sei					;enable all interrupts

		ldi		var,6	;Pointer auf "Sinus"
		mov		var_form,var
		ldi		var,2	;Pointer auf 10 Hz Stepweite per Encoderimpuls
		mov		var_step,var
		rcall	locval	;hole Wert entsprechend var_step
		lpm		RR0,z+	;dieser Wert bleibt ungendert
		lpm		RR1,z+	;solange im cmd_mode kein neuer Wert vorliegt
		lpm		RR2,z+	;diesen lt. je +- Drehimpuls zu RR[7:4] add/sub
		lpm		RR3,z
		rjmp	main							
; so, init ist getan

;************************************************************************
;*Pollin Panasonic 16348 Encoder mit 2ms Timerinterruptabfrage 
;* Autor: Hannes Lux (hannes) abgendert von  (rulixa)
;* http://www.mikrocontroller.net/topic/172127#1648272 
;* OLD NEW  Drehrichtung
;* 11  10   rechts			
;* 10  00   rechts
;* 00  01   rechts			
;* 01  11   rechts		
;*
;* 00  10   links			
;* 10  11   links		
;* 11  01   links			
;* 01  00   links = gegen Uhr 
;* neu, old, flag, B, C ,A		schon im Hauptprogramm definiert
;************************************************************************* 
Timer0: push	C 		;Timer0 ISR  ndert nur Register flag
		push	B		; "B" ist auch als "neu" definiert
		push	A
		push	ZL
		push	ZH
		sbi		pinD,6	;toggle zum abhren ob Interrupt luft, bzw wie
						;genau die Clockfrequenz ist, z.B.bei RC_oscillator
		in		C,sreg	;save Statusreg
		in		neu,PinD;Drehgeber-Port holen PD2 und PD3, Phase A, B
		andi	neu,0b00001100	;maskieren neu =PinD [3:2]
		lsr 	old		;old[3:2] wird old[1:0], "urold" wird Nirvana
		lsr		old		;von links kommt nur 0 nach
		or 		old,neu	;neue Drehgeberbits neu[3:2] nach old[3:2]
		;Drehrichtung lt. old aus Tabelle holen
		clr	A
		ldi	ZL,low (enctab*2)	;Tabelle mit Drehgeber-Increment-Werten
		ldi	ZH,high (enctab*2)
		add	ZL,old			;Index aufaddieren
		adc	ZH,A
		lpm	A,Z 			;Wert bei mir 0,1 oder 2
		tst	 flag			;ist flag von main schon abgeholt worden ?
		brne nnab
		mov	 flag,A			;neuen Wert speichern, egal welcher Wert
nnab:	out	 sreg,C			;restore Statusreg 
		pop	ZH
		pop	ZL
		pop	A
		pop	B
		pop	C
		reti
;wenn nicht gedreht wird, so ist neu = old, das kann sein 00 01 10 11
;bei jeder Kombi 0000 0101 1010 1111 steht 0 in der Tabelle = kein Auftrag								
;flag 1 oder 2 : Auftrag liegt vor
;wird der Auftrag in main ausgefhrt, so werden die Bits dort gelscht.

;Tabelle mit Drehgeber-Werten (neuneu-oldold als Index)
enctab:				;nn oo,     nn oo
.db     0, 0		;00 00,     00 01	;2 bytes je Zeile
.db     2, 0		;00 10,     00 11	;weil Assembler sowieso
.db     0, 0		;01 00,     01 01	;auf word auffllen wrde	
.db     0, 1 		;01 10,     01 11	;bisher verwendete Tabelle 
.db     1, 0		;10 00,     10 01	;	Adr_1=1 Adr_2=2		   
.db     0, 0		;10 10,     10 11	;	Adr13=2 Adr14=1
.db     0, 2		;11 00,     11 01	;hatte bei "rechts" Hnger bzw. 2fach
.db     0, 0		;11 10,     11 11


;.include "my_hlux.asm"			;Test welche Encodermethode
;.include "my_Timer0.asm"		;die wenigsten Fehler liefert
;.include "my_encoder_peda.asm"	;

;*************************************************************************
;* Jetzt folgt die Mainloop
;*************************************************************************
main:   sbis	PIND,4		;Taste gedrckt ?
		rjmp	cmd_mode	;ja, Kommandomodus

fmode1:	rcall	display_freq		;1. und 2. Zeile 
		cpi	flag,1		;bit0 gesetzt= Frequ. increment
		brne	eminus		;
		rcall	add32		;DDSaccu= DDSaccu + Frequenzdifferenz
		rjmp	disp_f

eminus:	cpi	flag,2			;bit1 gesetzt= Frequ. decrement
		brne	disp_f		;nix gltiges, auf Auftrag warten
		rcall	sub32		;DDSaccu= DDSaccu - Frequenzdifferenz		
		;bei flag = 0 passiert nix eben!

;nach encoder +- muss der DDS_controlteil den 32bit Phasenaccuwert senden
;der dann im DDS_uC die Frqeunz generiert

disp_f:		rcall	DDSaccu		;Accu holen, Frequenz ausrechnen,--> BCD 
		tst	flag
		BREQ	nosend5		;damit der DDS_uc nur bei neuen Daten
							;angesteuert wird
		rcall	send5		;sende Spannungsform und Wert an DDS-uC
nosend5:	clr	flag		;lsche tastenflag und warte auf Auftrag
		rcall	delay2ms
		rcall	delay2ms
		rcall	delay2ms 
		rcall	delay2ms
		rjmp	main		;
;*******************************************************************
; Kommandomode.  
; Wert in var einstellen durch Encoder, der 0...9 sein kann.
; var wird ausgehend vom alten var-Wert verndert
; 1. Der Text in Displayzeile 1 neu ausgegeben und
; 2. var wird de/incrementiert und der entsprechende Text im Display 	 
; 	 Zeile 2 ausgegeben und entspr. Wert nach RR[3:0] geschrieben
; 3.Nach Ende des cmd-mode hlt var den Pointer um wieviel Hertz pro
;	Impuls des Encoders die Frequenz gendert wird.
; Frequ.Step und Spannungsform also sin,sge,R-eck des DDS festlegen
; dazu werden die var Werte 0...9 ausgewertet, in Display Zeile 2 gezeigt
; ausgegeben und an den DDS_uC bermittelt
;********************************************************************
cmd_mode: sbic	PIND,4		;Taste gedrckt ?
		rjmp	update	;nein, Freq. mode

;Zeile1 und Zeile2 ausgeben		
		clr	flag	;soll nichts unerledigtes vom Frq_mode brig bleiben
		clr	ZH					;Text in Zeile1 schreiben
		ldi	ZL,_1cmd * 2		
		ldi	C,0					;Zeile1 Pos1
		rcall	lcd_flash_string;bentzt den Befehl lpm
		rcall	clrline2		;lsche 2. Zeile in cmd_mode
;Spannungsform in Zeile2 out
		ldi	ZL,_01		;Wordadresse von  Text = f (var)
		ldi	C,64		;setze Cursor 2.Zeile:Pos 11 
		rcall	locaddr2	;wird var_form bentzt
		rcall	lcd_flash_string
;Stepwerttext in Zeile2 out
		ldi	ZL,_01		;Wordadresse von  Text = f (var)
		ldi	C,74		;setze Cursor 2.Zeile:Pos 11 
		rcall	locaddr1	;wird var_step bentzt
		rcall	lcd_flash_string
		ldi	A,100
cmod:	rcall	delay2ms	;Display ist "ruhiger" mit 200ms delay
		dec	A
		brne	cmod
	

cmode1:	cpi	flag,1			;rechts  = Stepmode
		breq	incre		;Stepmode
		cpi		flag,2		;links   = Spgsformmode
		breq	incli		;Formmode
		rjmp	cmd_mode	;bleibe in cmd_mode, solange Taste gedrckt ist
							;in der ISR kann ja "gedreht" erkannt werden
incre:	clr	flag			;lsche tastenflag und fhre aus
		mov	var,var_step	;;Index fr Stepwert
		inc	var				;var setzt beim alten Wert fort
		cpi	var,6
		BRLO	spgs_form
		ldi	var,0			;Bereich 0....5
spgs_form:
		mov	var_step,var;Index fr Frequenzstep
		rcall	locval		;pointer auf Freq_step
		lpm	RR0,z+		;dieser Wert bleibt ungendert
		lpm	RR1,z+		;
		lpm	RR2,z+		;solange im cmd_mode kein neuer Wert vorliegt
		lpm	RR3,z		;dieser wird je +- Drehimpuls zu RR[7:4] add/sub
		rjmp	cmd_mode

incli:	clr	flag		;lsche tastenflag und fhre aus
		mov	var_form,var	;Index fr Spannungsformtext
		inc	var		;var setzt beim alten Wert fort
		cpi	var,6
		BRLO	zuniedrig
		cpi	var,10		;geht aber nur von 6...9
		brlo	u_form	
zuniedrig:	ldi	var,6

u_form:	mov	var_form,var
		rjmp	cmd_mode

update:	rcall	send5		; DDS_uC updaten
		rjmp	main
;Ende cmd_mode*********************************************

;Adresse rechen fr Displaytext 
;ZL=Source,  offset in var
;locaddr:push	B
;		clr	ZH	
;		mov	B,var			;mal 4, da 4 words Text je String
;		lsl	B
;		lsl	B			
;		add	ZL,B
;		lsl	ZL			; mal 2 fr Byteadresse
;		pop	B
;		ret					

;Adresse rechen fr Displaytext Zeile1
;ZL=Source,  offset in var_step
locaddr1:push	B
		clr	ZH	
		mov	B,var_step		;mal 4, da 4 words Text je String
		lsl	B
		lsl	B			
		add	ZL,B
		lsl	ZL			; mal 2 fr Byteadresse
		pop	B
		ret					
;Adresse rechen fr Displaytext Zeile2
;ZL=Source,  offset in var_form
locaddr2:push	B
		clr	ZH	
		mov	B,var_form		;mal 4, da 4 words Text je String
		lsl	B
		lsl	B			
		add	ZL,B
		lsl	ZL			; mal 2 fr Byteadresse
		pop	B
		ret					

;Adresse der Stepweitenwertes ausrechnen. df = f (var) nach RR[2:0]
;Input ist nur var, ZL=Wertetabelle gibts ja nur eine
locval:	push	B
		clr	ZH	
		ldi	ZL,z01		;Basisadresse der Stepwerte
		mov	B,var_step	;mal 2, da 2 words value je Step
		lsl	B		;aber nur 24 bit sind relevant
		add	ZL,B
		lsl	ZL		; mal 2 fr Byteadresse
		pop	B
		ret			; Z=Pointer auf Stepvalue

;-----------------------------------------------------------------
;send 5 Bytes zum DDS_uC via RS232 mit 500 kBaud
;Die Umschaltzeit auf neue Werte betrgt also nur 100 us.
;-----------------------------------------------------------------
send5:	cli
		push	A
		mov	A,var_form	;die Spannungsformtabelle im DDS_uC
		subi	A,5		;beginnt an 100er Bytegrenzen im Code
		rcall	WrCom		;das Spannungsformbyte ist gesendet
		mov	A,RR4		;byte1 des Phasenaccu
		rcall	WrCom		;out
		mov	A,RR5		;byte2	
		rcall	WrCom		;out
		mov	A,RR6		;byte3
		rcall	WrCom		;out
		mov	A,RR7		;byte4		
		rcall	WrCom		;alle 32 bit des Phasenaccus sind gesendet
		pop	A
		sei
		ret

WrCom:	sbis    UCSRA,UDRE          ;wait for UART ready
    	rjmp    WrCom
    	out     UDR,A
		ret
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                             ;;
;;       LCD-Routinen fr TC1602A Display      ;;
;;       =================================     ;;
;; LCD 4-bit mode SPLC780D standard controller ;;
;;    execution times INCLUDED in routines     ;;
;;                                             ;;		
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 
.equ LCD_PORTB = PORTB		;define the Ports / IOs
.equ LCD_DDRB  = DDRB
.equ PIN_RS   	=PD5		;Config for display RS und Enable
.equ PIN_E      =PD4


;send text pointed by source=Z und destination=C from flasch to LCD
lcd_flash_string:
           rcall	set_cur		;set display, DDRAM address
lcd_flash_1:
		lpm   	C, Z+		;pointed char nach C	   	
        cpi   	C, 00		;Stringende = 0
        breq  	lcd_flash_2
        rcall  	lcd_data
        rjmp  	lcd_flash_1
lcd_flash_2:ret


;Frequenz in Zeile 2 ab Pos. 10 darstellen 
display_freq:
;erste Zeile
		ldi	ZL,_1Zeile * 2		;flash in Words, hole aber Byte
		clr	ZH			;Display  Zeile1 F-mode ausgeben
		clr	C			;set source/dest. of string 
		rcall	lcd_flash_string	;bentzt den Befehl lpm

		ldi	C,10		;setze Cursor 1.Zeile:Pos 11 = DDRAM 10	
		ldi	ZL,_01
		rcall	locaddr1		
		rcall	lcd_flash_string

;zweite Zeile Spannungsform und Frequenz ausgeben
		clr	ZH
		ldi	ZL,_01
		rcall	locaddr2
		ldi	C,64			;prereqs fr nchste Programmzeile
		rcall	lcd_flash_string	;Spgsform 

;6 digits, "." und 1 digit ausgeben
		ldi	C,72
		rcall	set_cur		;im Anschlu an "Frequ." diese numerisch out
		clr	zh
		ldi	zl,23			;RX7 adressieren
		ld	C,z				;enthlt hi und lo nibble, hi verwerfen
		andi	C,15		;nur lo nibble ausgeben
		ori		C,$30
		rcall	lcd_data

		ldi 	B,2			;RX6 und RX5 ausgeben
		clr 	ZH			;Z auf RX6setzen
		ldi	ZL,23			;um 1 hher wegen "ld -Z" Befehl
bcdout:	ld   	C,-Z		;hoechste Stelle =r23= BCD8  zuerst raus
		swap 	C			;hi nibble holen
		andi 	C, 15		;ehemals hi nibble maskieren
		ori		C,$30		;in ASCII
		rcall 	lcd_data
		ld   	C,Z			;
		andi 	C, 15		;lo nibble detto
		ori		C,$30		;in ASCII
		rcall 	lcd_data
		dec 	B
		brne 	bcdout		;

		ld		C,-z			;jetzt RX4_hi, "." und RX4_lo ausgeben
		swap	C
		andi 	C, 15		;ehemals hi nibble maskieren
		ori		C,$30		;in ASCII
		rcall 	lcd_data
		ldi		C,'.'
		rcall 	lcd_data
		ld   	C,z			;
		andi 	C,15		;lo nibble detto
		ori		C,$30		;in ASCII
		rcall 	lcd_data			
		ret	
			
;send a data byte in C to the LCD, RS = high, changes C
lcd_data:
		push	C				;copy the data byte for later
		swap	C 				;hi nibble zuerst senden
		andi	C, 0b00001111	;clear ehemals low nibble 
		sbr		C, 1<<PIN_RS	;es sind Daten RS=1
		out		LCD_PORTB, C 	;prepare LCD port for write
		rcall 	lcd_enable		;LCD enable	writes now + 50us

		pop	C				;process now low nibble 
		andi	C, 0b00001111 
		sbr		C, 1<<PIN_RS 	;es sind Daten RS=1   	
		out		LCD_PORTB, C	;
		rcall	lcd_enable 
		ret                          
 
;send a command in C to the LCD, RS = low	changes C
;bis auf RS kein Unterschied zu lcd_data
lcd_cmd:push	C          		;copy the command byte for later
		swap	C				;hi nibble zuerst senden
		andi	C, 0b00001111  	;clear ehemals low nibble und RS=0
		out		LCD_PORTB, C	;prepare LCD port for write
		rcall 	lcd_enable		;LCD enable	writes now + 50us
	
		pop		C				;process now low nibble 
		andi	C, 0b00001111	;RS ist damit automatisch 0
		out		LCD_PORTB, C       	;
		rcall	lcd_enable 
		ret                                                      
  
clrline2:ldi	C,64	;lsche Zeile2 des Displays
		rcall	set_cur
		ldi		A,15
clr1:	ldi		C,' '
		rcall	lcd_data
		dec		A
		brne	clr1
		ret
		         
;create  enable pulse 5 cycles auf 1 macht 0,75 us ndert kein Register
;mit fallender Flanke werden Daten bernommen
lcd_enable:
       sbi LCD_PORTB, PIN_E         ;2; Enable high
       nop
       nop
       nop
       cbi LCD_PORTB, PIN_E         ;2; Enable low
	   rcall	delay50us		;ab H>L Enable braucht das Display 38 us
	   ret				;bis es wieder bereit ist
 
; set cursor to DDRAM position in C, changes C
; 1. line [C]= 0.....15d
; 2. line [C]= 64d...79d
set_cur:ori	C, $80	;Befehl selbst
		rcall	lcd_cmd
		rcall	delay50us	;50 us sollte an sich gengen
		ret                        
 
; > 38 us delay,	ndert kein Register
delay50us:push	A
		ldi	A, 96
delay50us_:dec	A		;(5*96-1+7)*0,125 = 52,8 us
		nop
		nop
        brne	delay50us_
		pop		A
		ret                          
 
; > 1,52 ms delay for lcd_clr und lcd_home	ndert kein Register
delay2ms:push	C
		push	B
		ldi 	C, 32
w2ms0:	ldi		B, 200
w2ms1:	dec		B		;delay 3*R17 -1 Takte ~75 uS
		brne	w2ms1
		dec		C		;+1
		brne	w2ms0		;(600*32 +10)*0,125
		pop		B
		pop		C
		ret  			;= 2401 uS

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Ende der Displayroutinen, Beginn Mathematik ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;--------------------------------------------------------------------
;-- RR[7:4] = RR[7:4] - RR[3:0]	
;--------------------------------------------------------------------
sub32:	sub	RR4,RR0
		sbc	RR5,RR1
		sbc	RR6,RR2
		sbc	RR7,RR3
		mov	A,RR7
		cpi	A,0x13	;Frequenz < 0 geworden, auf max. setzen
		brlo	gut
		clr	RR4
		clr	RR5
		clr	RR6
		ldi	A, 0x12
		mov	RR7,A		;vom maxwert runterzhlen
gut:	ret

;--------------------------------------------------------------------
;-- RR[7:4] = RR[7:4] + RR[3:0]	
;--------------------------------------------------------------------
add32:	add	RR4,RR0
		adc	RR5,RR1
		adc	RR6,RR2
		adc	RR7,RR3
		mov	A,RR7
		cpi	A,0x12	;Frequenz zu hoch
		brlo	gut1
		clr	RR4
		clr	RR5
		clr	RR6
		clr	RR7
		inc	RR4		;auf 0,0000338 Hz setzen
gut1:		ret

;-----------------------------------------------------------------------
; die 4 Byte vom DDS Teil als 7 digit Ascii auf Display Zeile 2 out
; RX[7:0] = 64bit result of RX[3:0] * RY[3:0]
;-----------------------------------------------------------------------
		;RR[7:4] ist der Phasenaccu
		;RY[3:0] wird in der Init aktiviert

DDSaccu:mov	RX0,RR4		;transfer DDSaccu 
		mov	RX1,RR5
		mov	RX2,RR6		;um die
		mov	RX3,RR7
		rcall 	mul32u		;die Frequenz zu berechnen und in
		rcall	Bin3BCD16	;BCD ausgeben
		ret
 
 ;*********************************************************************
;* Frequenz berechnen mittels 32 Bit Multiplikation und 
;* anschlieender Division durch 2^32.  f = DDSaccu * clock/11 / 2^32
;* beim DDS mit 32 Bit Phasenaccu reichen an sich 24 Bits da max.
;* 1022727 1/10 Hz Schritte im Display angezeigt werden, von 1,67e6 mgl.
;**********************************************************************
;  Call:  RX[3:0] = 32bit multiplicand
;         RX[7:4] = <don't care>
;         RY[3:0] = 32bit multiplier  (clock/11 *10) wegen 1/10 Hz units
;         A        = <don't care> (high register must be allocated)
;  Result:RX[7:0] = 64bit result of RX[3:0] * RY[3:0]
;         RY[3:0] = <not changed>
;         A       = unchanged
;nach der 2^32 Division steht Result in RX[3:0]
;
; Size  = 21 words
; Clock = 436..532 cycles (+ret)  72 uS bei 8 MHz
; Stack = 0 byte

mul32u:	push	A
		sub	RX7,RX7	
		sub	RX6,RX6
		sub	RX5,RX5
		sub	RX4,RX4		;initialize variables
		ldi	A,33		; 
		brcc	PC+5		; calculating loop
		add	RX4,RY0		;
		adc	RX5,RY1		;
		adc	RX6,RY2	
		adc	RX7,RY3
		ror	RX7
		ror	RX6		;
		ror	RX5		;ins Ergebnis schieben
		ror	RX4		;
		ror	RX3		;
		ror	RX2		;
		ror	RX1		;
		ror	RX0		;
		dec	A		;bis alle Bits durchgearbeitet sind
		brne	PC-14		;continue loop;
;so, jetzt durch 2^32 dividieren
div2_32:mov	RX3,RX7	
		mov	RX2,RX6		;divide by 2^32 ist ident mit Komma 4x nach links
		mov	RX1,RX5
		mov	RX0,RX4
		pop	A
		ret			;das war's auch schon

;***************************************************************************
;* Bin3BCD == 24-bit Binary to BCD conversion
;*
;*  RX0:RX1:RX2  >>>  RX4:RX5:RX6:RX7 es gibt max. 8 digits!
;*	  hex			     dec
;* see AVR200 bis MATH32X application notes, http://avr-asm.tripod.com
;* sehr gute Seite ist auch http://elm-chan.org u.a. Wurzel ziehen
;***************************************************************************
;.def	RX0	=r16	; binary value byte 0 (LSB)
;.def	RX1	=r17	; binary value byte 1
;.def	RX2	=r18	; binary value byte 2 
;.def	RX3	=r19	; binary value byte 3 (MSB)
;.def	RX4	=r20	; BCD value digits 0 and 1
;.def	RX5	=r21	; BCD value digits 2 and 3
;.def	RX6	=r22	; BCD value digits 4 and 5
;.def	RX7	=r23	; BCD value digits 6 and 7 (MSD)

Bin3BCD16:	ldi	RX7,0xfa		;initialize digits 7 and 6
binbcd_107:	subi	RX7,-0x10		;
		subi	RX0,byte1(10000*1000) ;subit fbin,10^7
		sbci	RX1,byte2(10000*1000) ;
		sbci	RX2,byte3(10000*1000) ;
		brcc	binbcd_107		;
binbcd_106:	dec	RX7			;
		subi	RX0,byte1(-10000*100) ;addit fbin,10^6
		sbci	RX1,byte2(-10000*100) ;kann ja schon zuviel subtr. sein
		sbci	RX2,byte3(-10000*100) ;wieder dazuaddieren
		brcs	binbcd_106		;
		ldi	RX6,0xfa		;initialize digits 5 and 4
binbcd_105:	subi	RX6,-0x10		;
		subi	RX0,byte1(10000*10)	;subit fbin,10^5
		sbci	RX1,byte2(10000*10)	;
		sbci	RX2,byte3(10000*10)	;
		brcc	binbcd_105		;
binbcd_104:	dec	RX6			;
		subi	RX0,byte1(-10000)	;addit fbin,10^4
		sbci	RX1,byte2(-10000)	;
		sbci	RX2,byte3(-10000)	;
		brcs	binbcd_104		;
		ldi	RX5,0xfa		;initialize digits 3 and 2
binbcd_103:	subi	RX5,-0x10		;
		subi	RX0,byte1(1000)	;subiw fbin,10^3
		sbci	RX1,byte2(1000)	;
		brcc	binbcd_103		;
binbcd_102:	dec	RX5			;
		subi	RX0,byte1(-100)	;addiw fbin,10^2
		sbci	RX1,byte2(-100)	;
		brcs	binbcd_102		;
		ldi	RX4,0xfa		;initialize digits 1 and 0
binbcd_101:
		subi	RX4,-0x10		;
		subi	RX0,10		;subi fbin,10^1
		brcc	binbcd_101		;
		add		RX4,RX0		;LSD
		ret				;

.dseg


